/* -*- mode: c -*- */
/*****************************************************************************
  ls11main.c : Main implementation of the KOEI LS file manipulation program.
  numdisp, 2010-04-10, rev 0.
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ls11core.h"

/* 32-bit unsigned integer (4 bytes). */
typedef unsigned int  UINT32;

#define LS11_ERR_NONE                0
#define LS11_ERR_OPEN_INPUT_FAILED   1
#define LS11_ERR_OPEN_OUTPUT_FAILED  2
#define LS11_ERR_INVALID_FILE_TYPE   3
#define LS11_ERROR_DECODE_FAILED     4
#define LS11_ERR_NOT_ENOUGH_MEMORY   5

/****************************************************************************/
/* Error processing of the program. */

static int nErr =  LS11_ERR_NONE;
static char *szErr[] = {
    "OK.",
    "Cannot open the encoded file!",
    "Cannot write the decoded file!",
    "Unsupported encoded file type!",
    "Failed to decode the file!",
    "Not enough memory!"
};

static void set_last_error(int err)
{
    nErr = err;
}

static int get_last_error()
{
    return nErr;
}

/****************************************************************************/

/* Decode an LS encoded file.*/
static int ls11_file_decode(const char *fnenc, const char *fndec)
{
    FILE *fpenc, *fpdec;
    unsigned char dict[LS_DICT_LENGTH]; /* The dictionary. */
    unsigned char *encs = NULL;         /* The encoded stream. */
    unsigned char *decs = NULL;         /* The decoded stream. */
    UINT32 *info[3], inftmp[3];         /* The information blocks. */
    int ninfs, maxlenc, maxldec, lenc, ldec, lstype, i;
    int lenctotal = 0, ldectotal = 0;
    long pos;

    if ((fpenc = fopen(fnenc, "rb")) == NULL) {
        set_last_error(LS11_ERR_OPEN_INPUT_FAILED);
        return 0;
    }

    lstype = get_ls_file_type(fpenc);
    if (lstype == INVALID_LSTYPE) {
        fclose(fpenc);
        set_last_error(LS11_ERR_INVALID_FILE_TYPE);
        return 0;
    }

    if ((fpdec = fopen(fndec, "wb")) == NULL) {
        fclose(fpenc);
        set_last_error(LS11_ERR_OPEN_OUTPUT_FAILED);
        return 0;
    }

    /* Get the dictionary. */
    fseek(fpenc, LS_HEAD_LENGTH, SEEK_SET);
    fread(dict, LS_DICT_LENGTH, 1, fpenc);

    /* Peek the data from the first encoding information section. */
    fread(inftmp, 4 * 3, 1, fpenc);
    swapednv(&inftmp, 4, 3);
    maxlenc = inftmp[0];
    maxldec = inftmp[1];
    ninfs = (inftmp[2] - 4 - LS_HEAD_LENGTH - LS_DICT_LENGTH) / (4 * 3);

    /* Get the encoding information. */
    info[0] = (UINT32*) malloc(ninfs * 4 * 3);
    if (info[0] == NULL) {
        fclose(fpenc);
        fclose(fpdec);
        set_last_error(LS11_ERR_NOT_ENOUGH_MEMORY);
        return 0;
    }
    info[1] = info[0] + ninfs;
    info[2] = info[1] + ninfs;
    fseek(fpenc, LS_HEAD_LENGTH + LS_DICT_LENGTH, SEEK_SET);
    for (i = 0; i < ninfs; ++i) {
        fread(&info[0][i], 4, 1, fpenc);
        fread(&info[1][i], 4, 1, fpenc);
        fread(&info[2][i], 4, 1, fpenc);
        swapednv(&info[0][i], 4, 1);
        swapednv(&info[1][i], 4, 1);
        swapednv(&info[2][i], 4, 1);
        if (maxlenc < ((int) info[0][i])) {
            maxlenc = info[0][i];
        }
        if (maxldec < ((int) info[1][i])) {
            maxldec = info[1][i];
        }
        lenctotal += info[0][i];
        ldectotal += info[1][i];
    }

    /* Double check the file format. */
    fseek(fpenc, 0, SEEK_END);
    pos = ftell(fpenc);
    pos -= LS_HEAD_LENGTH + LS_DICT_LENGTH + 4 + ninfs * 4 * 3;
    if (pos != ((long)lenctotal)) {
        free(info[0]);
        fclose(fpenc);
        fclose(fpdec);
        set_last_error(LS11_ERR_INVALID_FILE_TYPE);
        return 0;
    }

    /* Allocate the memory for the encoding and decoding streams. */
    encs = (unsigned char*) malloc(maxlenc);
    if (encs == NULL) {
        free(info[0]);
        fclose(fpenc);
        fclose(fpdec);
        set_last_error(LS11_ERR_NOT_ENOUGH_MEMORY);
        return 0;
    }
    decs = (unsigned char*) malloc(maxldec);
    if (decs == NULL) {
        free(encs);
        free(info[0]);
        fclose(fpenc);
        fclose(fpdec);
        set_last_error(LS11_ERR_NOT_ENOUGH_MEMORY);
        return 0;
    }

    /* Do the decoding. */
    for (i = 0; i < ninfs; ++i) {
        lenc = info[0][i];
        ldec = info[1][i];
        pos = (long)info[2][i];
        fseek(fpenc, pos, SEEK_SET);
        fread(encs, lenc, 1, fpenc);
        if (!ls11_decode(dict, encs, lenc, decs, ldec, lstype)) {
            free(decs);
            free(encs);
            free(info[0]);
            fclose(fpenc);
            fclose(fpdec);
            set_last_error(LS11_ERROR_DECODE_FAILED);
            return 0;
        }
        fwrite(decs, ldec, 1, fpdec);
    }

    free(decs);
    free(encs);
    free(info[0]);
    fclose(fpenc);
    fclose(fpdec);
    return 1;
}

/****************************************************************************/

int main(int argc, char *argv[])
{
    int n, n1, i;
    char *fnout;

    /* Check the usage. */
    if (argc < 2) {
        printf("LS11 decoding utility, version 0.60\n");
        printf("By numdisp, 2010-04-10\n");
        printf("Usage: %s FILE ...\n", argv[0]);
        return 1;
    }

    /* Allocate the memory for output filenames. */
    n = (int) strlen(argv[1]);
    for (i = 2; i < argc; ++i) {
        n1 = (int) strlen(argv[i]);
        if (n < n1) {
            n = n1;
        }
    }
    fnout = (char*) malloc(n + 16);

    /* Process the targets one by one. */
    n = 0;
    for (i = 1; i < argc; ++i) {
        set_last_error(LS11_ERR_NONE);
        printf("%s : ", argv[i]);
        strcpy(fnout, argv[i]);
        strcat(fnout, ".DEC");
        if (ls11_file_decode(argv[i], fnout)) {
            ++n;
        }
        printf("%s\n", szErr[get_last_error()]);
    }

    /* All done. */
    free(fnout);
    printf("\nTotal targets: %d, ", argc - 1);
    printf("Succeeded: %d, ", n);
    printf("Failed: %d\n", argc - 1 - n);

    return 0;
}

/*****************************************************************************
  End of the C source file : ls11main.c
*****************************************************************************/
