/* -*- mode: c -*- */
/*****************************************************************************
  ls11core.c : Implementation of the KOEI LS file manipulation functions.
  numdisp, 2010-04-10, rev 0.
*****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "ls11core.h"

/*============================================================================
  [Function]
    Swaps the endianness (byte order) of a vector.
  [Parameters]
    x    : The starting address of the vector to be converted.
    size : Size of each element of the vector in bytes.
    num  : Number of elements to be converted in the vector.
  [Return Values]
    None.
  [Remarks]
    Elements of the vector must be stored in contiguous memory locations.
    If the parameter num is greater than the actual size (total number of
    elements) of the vector, the function behavior is undefined.
----------------------------------------------------------------------------*/
void swapednv(void *x, unsigned int size, unsigned int num)
{
    unsigned char *p, *p1, *p2, b;
    unsigned int s2, n, i;

    assert(x != NULL && size > 1 && num > 0);

    s2 = size / 2;
    p = (unsigned char*) x;
    for (n = 0; n < num; ++n) {
        for (i = 0; i < s2; ++i) {
            p1 = p + i;
            p2 = p + 2 * s2 - i - 1;
            /* *p1 ^= *p2 ^= *p1 ^= *p2; */
            b = *p1;
            *p1 = *p2;
            *p2 = b;
        }
        p += size;
    }
}

/*============================================================================
  [Function]
    Get the detailed type of a LS encoded file.
  [Parameters]
    fp : Target FILE structure.
  [Return Values]
    LS10TYPE for LS10 files, LS11TYPE for LS11 files, LS12TYPE for LS12 files,
    or INVALID_LSTYPE if it is not a valid LS file.
  [Remarks]
    None.
----------------------------------------------------------------------------*/
int get_ls_file_type(FILE *fp)
{
    long pos0;
    char head[LS_HEAD_LENGTH] = { 0 };  /* The standard file header. */
    char buf[LS_HEAD_LENGTH];           /* File header to be checked. */
    int type = LS11TYPE;                /* The file type number. */
    int i;

    assert(fp != NULL);

    /* Backup the original file pointer position. */
    pos0 = ftell(fp);

    /* Set the standard file head. */
    head[0] = 'L';
    head[1] = 'S';
    head[2] = '1';
    head[3] = '1';

    /* Read the header information from the file. */
    fseek(fp, 0, SEEK_SET);
    if (fread(buf, LS_HEAD_LENGTH, 1, fp) != 1) {
        return INVALID_LSTYPE;
    }

    /* Compare the file header with the standard header. */
    for (i = 0; i < LS_HEAD_LENGTH; ++i) {
        if (buf[i] != head[i]) {
            if (i == 3) {
                if (buf[i] == '0') {
                    type = LS10TYPE;
                    continue; /* The next loop. */
                } else if (buf[i] == '2') {
                    type =LS12TYPE;
                    continue; /* The next loop. */
                }
            }
            return INVALID_LSTYPE;
        }
    }

    /* Restore the original file pointer position. */
    fseek(fp, pos0, SEEK_SET);

    return type;
}

/*============================================================================
  [Function]
    Decode an LS encoded byte stream.
  [Parameters]
    dict   : The code dictionary.
    enc    : The encoded byte stream.
    lenc   : Length of the encoded byte stream.
    dec    : The byte stream after decoding.
    ldec   : Length of the stream after decoding.
    lstype : Type of the encoded algorithm.
  [Return Values]
    1 if the encoded stream is successfully decoded, otherwise 0.
  [Remarks]
    The length of the dictionary must be exactly LS_DICT_LENGTH.
    All the streams must be not NULL and have length of at least 1. Otherwise
    the function behavior is undefined.
    lstype must be LS10TYPE, LS11TYPE, or LS12TYPE. Otherwise the function
    will return immediately without doing anything.
----------------------------------------------------------------------------*/
int ls11_decode(unsigned char *dict, unsigned char *enc, int lenc,
                unsigned char *dec, int ldec, int lstype)
{
    unsigned int code, mask, bitlen, bitset;
    int bitpos = 7; /* Index of the current processing bit. */
    int copytail = 3; /* The number that add to the redundant copies.*/
    int back = 0; /* Bytes need to move to the start position of copy. */

    /* The variable "lstype" is unused in current implementation. This line
    is just for avoiding the "unused variable" warning by some compilers. */
    lstype = lstype;

    if (lenc == ldec) { /* No compression, so just copy and return. */
        memcpy(dec, enc, lenc);
        return 1;
    }

    #define GETBIT(_byte, _pos)  ((_byte >> _pos) & 0x01)

    /* Start the decoding. */
    while (ldec > 0) {
        assert(lenc > 0);

        /************************************************/
        /* Get the code from the bit stream. */

        /* Get the first part of the code. */
        code = bitlen = 0;
        do {
            bitset = GETBIT(*enc, bitpos);
            code = (code << 1) | bitset;
            ++bitlen;
            --bitpos;
            if (bitpos < 0) {
                bitpos = 7;
                ++enc;
                --lenc;
            }
        } while (bitset);

        /* Now get the mask of the code. */
        mask = 0;
        while (bitlen--) {
            bitset = GETBIT(*enc, bitpos);
            mask = (mask << 1) | bitset;
            --bitpos;
            if (bitpos < 0) {
                bitpos = 7;
                ++enc;
                --lenc;
            }
        }

        /* Get the final code by combining the first part with the mask. */
        code += mask;

        /************************************************/

        if (back == 0 && code >= LS_DICT_LENGTH) {
            back = code - LS_DICT_LENGTH;
            /* Notice that the value of the code should not equal to the
            length of the dictionary. Otherwise the value of variable "back"
            will be zero, which leads the redundant copy to start from an
            undefined position. */
            assert(back != 0);
            if (back == 0) {
                return 0;
            }
            continue; /* Get the next code. */
        }

        if (back == 0) {
            assert(code < LS_DICT_LENGTH);
            if (code >= LS_DICT_LENGTH) {
                return 0;
            }
            /* Just get the item from the dictionary. */
            *dec = *(dict + code);
            ++dec;
            --ldec;
        } else {
            /* Copy the items from the previously decoded stream. */
            code += copytail;
            while (code--) {
                *dec = *(dec - back);
                ++dec;
                --ldec;
            }
            back = 0;
        }
    }

    /* Double check the length information to tell if the original LS stream
    is properly decoded. If everything is OK, "ldec" should reduced to zero,
    "lenc" should be a number close to or equal to zero. */
    return lenc >= 0 && ldec == 0 ? 1 : 0;
}

/*****************************************************************************
  End of the C source file : ls11core.c
*****************************************************************************/
